/**
 * \file: mspin_connection_tls_server.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * mySPIN Connection Adapter TCP/IP
 *
 * \component: MSPIN
 *
 * \author: Thilo Bjoern Fickel BSOT/PJ-ES1 thilo.fickel@bosch-softtec.com
 *
 * @copyright: (c) 2016 Bosch SoftTec GmbH
 *
 * \history
 * 0.1 TFickel Initial version
 *
 ***********************************************************************/

#include "mspin_connection_tcp_server.h"
#include "mspin_connection_adapter.h"
#include "mspin_connection_tcp_manager.h"
#include "mspin_connection_tcp_listener.h"

#include "mspin_logging.h"

#include <pthread.h>        //pthread_*
#include <string.h>         //memset, strerror
#include <sys/socket.h>     //socket
#include <arpa/inet.h>
#include <netinet/udp.h>    //udp
#include <errno.h>          //errno
#include <ifaddrs.h>        //ifaddrs
#include <sys/prctl.h>      //prctl
#include <stdio.h>          //snprintf

MSPIN_ERROR mspin_tcp_connectClient(mspin_context_t* pContext, S32 socketFD)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, connectionID=%d) entered", __FUNCTION__, pContext, socketFD);

    if (pContext)
    {
        mspin_connectionHandle_t *pConnectionHandle = NULL;

        //Release connection handles
        mspin_conn_releaseHandle(&(pContext->pConnectionHandle));

        //Create connection handle
        pConnectionHandle = mspin_conn_createHandle();
        if (!pConnectionHandle)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, id=%d) ERROR: Failed to create connection handle",
                 __FUNCTION__, pContext, socketFD);

            mspin_tcp_closeConnection(socketFD);
            mspin_tcp_signalConnectionClosed(socketFD, mspin_tcp_getIPAddr(socketFD), MSPIN_TCP_CONNECTION_OTHER_ERROR);

            return MSPIN_ERROR_MEM_ALLOC;
        }

        //Assign values
        pConnectionHandle->connectionType = MSPIN_CONNECTION_SERVER_TCP;
        pConnectionHandle->result = 0; //success
        pConnectionHandle->connectionID = socketFD;

        pContext->pConnectionHandle = pConnectionHandle;

        if (!mspin_tcp_setActive(socketFD))
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, id=%d) ERROR: Could not find connection ID",
                 __FUNCTION__, pContext, socketFD);

            mspin_tcp_closeConnection(socketFD);
            mspin_tcp_signalConnectionClosed(socketFD, mspin_tcp_getIPAddr(socketFD), MSPIN_TCP_CONNECTION_OTHER_ERROR);

            return MSPIN_ERROR_INVALID_PARAMETER;
        }

        //For a later implementation: Add creation of message queue etc. here (if used)

        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, id=%d) done using connectionHandle=%p",
                __FUNCTION__, pContext, socketFD, pConnectionHandle);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, socketFD=%d) ERROR: Connection handle is NULL",
                __FUNCTION__, pContext, socketFD);

        return MSPIN_ERROR_NULL_HANDLE;
    }

    return MSPIN_SUCCESS;
}

MSPIN_ERROR mspin_tcp_disconnectClient(mspin_context_t* pContext)
{
    if (pContext && pContext->pConnectionHandle)
    {
        in_addr_t ipAddr = 0;
        MSPIN_ERROR result = MSPIN_SUCCESS;
        int connectionID = pContext->pConnectionHandle->connectionID;

        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) entered using connID=%d",
                __FUNCTION__, pContext, connectionID);

        ipAddr = mspin_tcp_getIPAddr(connectionID);

        //Release the connection
        result = mspin_tcp_closeConnection(connectionID);
        if (MSPIN_SUCCESS == result) //if found
        {
            mspin_tcp_signalConnectionClosed(connectionID, ipAddr, MSPIN_TCP_CONNECTION_MYSPIN_SESSION_ENDED);
        }

        pContext->pConnectionHandle->connectionID = -1; //invalidate connection ID regardless of result

        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) connection closed",
                __FUNCTION__, pContext);

        return result;
    }
    else if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: Context is NULL",
                __FUNCTION__, pContext);
        return MSPIN_ERROR_NULL_HANDLE;
    }
    else if (!pContext->pConnectionHandle)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: Connection handle is NULL",
                __FUNCTION__, pContext);
        return MSPIN_ERROR_NULL_HANDLE;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: TCP handle is NULL",
                __FUNCTION__, pContext);
        return MSPIN_ERROR_NULL_HANDLE;
    }
}
